import { PrismaClient, Prisma } from '@prisma/client';
import { ZipCodeRepository } from '../repository/zipcode.repository.js';
export class LocationsService {
    prisma;
    zipCodeRepository;
    constructor() {
        this.prisma = new PrismaClient();
        this.zipCodeRepository = new ZipCodeRepository();
    }
    async getLatLongByZipCode(zip) {
        return this.zipCodeRepository.getLatLongByZip(zip);
    }
    async getAllLocations(limit = 50, offset = 0) {
        // Get the total count
        const totalCount = await this.prisma.location.count();
        // Get paginated locations
        const locations = await this.prisma.location.findMany({
            include: {
                brand: true,
                storeAttributes: {
                    include: {
                        submittedBy: {
                            select: {
                                id: true,
                                name: true,
                                email: true,
                            },
                        },
                        allocationMethods: true,
                        votes: true,
                    },
                    orderBy: {
                        createdAt: 'desc',
                    },
                    take: 1, // Get the most recent attribute for each location
                },
            },
            take: limit,
            skip: offset,
            orderBy: {
                name: 'asc',
            },
        });
        return {
            data: locations,
            totalCount,
        };
    }
    async getLocationById(id) {
        return this.prisma.location.findUnique({
            where: { id },
            include: {
                brand: true,
                storeAttributes: {
                    include: {
                        submittedBy: {
                            select: {
                                id: true,
                                name: true,
                                email: true,
                            },
                        },
                        allocationMethods: true,
                        votes: true,
                    },
                    orderBy: {
                        createdAt: 'desc',
                    },
                },
            },
        });
    }
    /**
     * Search locations with various filters and by proximity to zip code
     * Uses PostgreSQL's geographic features for efficient distance calculation
     */
    async searchLocations(params) {
        const { query = '', zipCode, type, hasStorePicks, offersSamples, sortBy = 'name', sortOrder = 'asc', limit = 50, offset = 0, maxDistance, latitude, longitude, } = params;
        // Build the base query
        let whereClause = {};
        let coordinates = null;
        // Text search on name, address, city, or state
        if (query) {
            whereClause.OR = [
                { name: { contains: query, mode: 'insensitive' } },
                { city: { contains: query, mode: 'insensitive' } },
                { address: { contains: query, mode: 'insensitive' } },
                { state: { contains: query, mode: 'insensitive' } },
            ];
        }
        // Filter by location type
        if (type) {
            whereClause.type = type;
        }
        // Determine coordinates from latitude/longitude or zip code
        if (typeof latitude === 'number' && typeof longitude === 'number') {
            coordinates = { lat: latitude, lng: longitude };
        }
        else if (zipCode) {
            try {
                // Use the zipcode service to get the coordinates
                const zipData = await import('../services/zipcode.service.js');
                const zipCodeData = await zipData.getZip(zipCode);
                if (zipCodeData) {
                    coordinates = {
                        lat: Number(zipCodeData.latitude),
                        lng: Number(zipCodeData.longitude),
                    };
                }
                else {
                    console.error('Zip code not found in database:', zipCode);
                }
            }
            catch (error) {
                console.error('Error fetching zip code coordinates:', error);
                // Continue with the query without coordinates
            }
        }
        // Get total count for pagination
        const totalCount = await this.prisma.location.count({
            where: whereClause,
        });
        // Different query approaches based on whether we're doing a distance-based search
        let locations = [];
        // If zipCode was provided but coordinates couldn't be determined, return empty result
        if (zipCode && !coordinates) {
            return {
                data: [],
                totalCount: 0,
            };
        }
        if (coordinates) {
            try {
                // Add distance calculation using PostgreSQL's earth_distance or PostGIS
                // For PostgreSQL without PostGIS, we use the Haversine formula directly in SQL
                const queryResults = await this.prisma.$queryRaw `
          SELECT 
            l.*,
            (
              3958.8 * acos(
                LEAST(1.0, cos(radians(${coordinates.lat})) * 
                cos(radians(CAST(l.latitude AS float))) * 
                cos(radians(CAST(l.longitude AS float)) - radians(${coordinates.lng})) + 
                sin(radians(${coordinates.lat})) * 
                sin(radians(CAST(l.latitude AS float))))
              )
            ) as distance
          FROM "xplr"."locations" l
          WHERE 
            l.latitude IS NOT NULL 
            AND l.longitude IS NOT NULL
            ${type ? Prisma.sql `AND l.type::text = ${type}` : Prisma.sql ``}
            ${query
                    ? Prisma.sql `AND (
                l.name ILIKE ${'%' + query + '%'} OR
                l.city ILIKE ${'%' + query + '%'} OR
                l.address ILIKE ${'%' + query + '%'} OR
                l.state ILIKE ${'%' + query + '%'}
              )`
                    : Prisma.sql ``}
          ORDER BY distance ASC
          LIMIT ${limit} OFFSET ${offset}
        `;
                // Store the raw query results with distance information
                const rawLocationsWithDistance = queryResults.map((loc) => ({
                    ...loc,
                    distance: parseFloat(loc.distance),
                }));
                // Now fetch related data for these locations in a single query
                if (rawLocationsWithDistance.length > 0) {
                    // Get location IDs for the IN clause
                    const locationIds = rawLocationsWithDistance.map((loc) => loc.id);
                    // Fetch related data for the locations
                    const locationsWithRelations = await this.prisma.location.findMany({
                        where: {
                            id: {
                                in: locationIds,
                            },
                        },
                        include: {
                            brand: true,
                            storeAttributes: {
                                include: {
                                    submittedBy: {
                                        select: {
                                            id: true,
                                            name: true,
                                            email: true,
                                        },
                                    },
                                    allocationMethods: true,
                                    votes: true,
                                },
                                orderBy: {
                                    createdAt: 'desc',
                                },
                                take: 1, // Get only the most recent store attributes
                            },
                        },
                    });
                    // Create a map for faster lookups
                    const relationsMap = new Map();
                    locationsWithRelations.forEach((loc) => {
                        relationsMap.set(loc.id, {
                            brand: loc.brand,
                            storeAttributes: loc.storeAttributes,
                        });
                    });
                    // Combine distance data with relations
                    locations = rawLocationsWithDistance.map((loc) => {
                        const relations = relationsMap.get(loc.id) || { brand: null, storeAttributes: [] };
                        return {
                            ...loc,
                            brand: relations.brand,
                            storeAttributes: relations.storeAttributes,
                        };
                    });
                }
                else {
                    locations = [];
                }
                // Sort by distance if requested
                if (sortBy === 'distance') {
                    locations.sort((a, b) => (a.distance || Infinity) - (b.distance || Infinity));
                }
            }
            catch (error) {
                console.error('Error executing distance-based location search:', error);
                throw new Error('Failed to search locations by distance');
            }
        }
        else {
            // Regular query without distance calculation
            locations = await this.prisma.location.findMany({
                where: whereClause,
                include: {
                    brand: true,
                    storeAttributes: {
                        include: {
                            submittedBy: {
                                select: {
                                    id: true,
                                    name: true,
                                    email: true,
                                },
                            },
                            allocationMethods: true,
                            votes: true,
                        },
                        orderBy: {
                            createdAt: 'desc',
                        },
                        take: 1,
                    },
                },
                orderBy: {
                    [sortBy]: sortOrder,
                },
                take: limit,
                skip: offset,
            });
        }
        // Apply additional filters for store attributes
        let filteredLocations = locations;
        if (hasStorePicks !== undefined || offersSamples !== undefined) {
            filteredLocations = locations.filter((location) => {
                if (!location.storeAttributes || location.storeAttributes.length === 0) {
                    return false;
                }
                const attrs = location.storeAttributes[0];
                if (hasStorePicks !== undefined && attrs.hasStorePicks !== hasStorePicks) {
                    return false;
                }
                if (offersSamples !== undefined && attrs.offersSamples !== offersSamples) {
                    return false;
                }
                return true;
            });
            // Adjust totalCount for filtered results
            if (filteredLocations.length !== locations.length) {
                return {
                    data: filteredLocations,
                    totalCount: filteredLocations.length,
                };
            }
        }
        return {
            data: filteredLocations,
            totalCount,
        };
    }
    async searchLocationsByZipCode(zipCode, maxDistance) {
        // Get coordinates for the zip code using zipcode service
        let coordinates = null;
        try {
            // Use the zipcode service to get the coordinates
            const zipData = await import('../services/zipcode.service.js');
            const zipCodeData = await zipData.getZip(zipCode);
            if (zipCodeData) {
                coordinates = {
                    lat: Number(zipCodeData.latitude),
                    lng: Number(zipCodeData.longitude),
                };
            }
        }
        catch (error) {
            console.error('Error fetching zip code coordinates:', error);
            throw new Error('Invalid zip code or unable to fetch coordinates');
        }
        if (!coordinates) {
            throw new Error('Unable to determine coordinates for the provided zip code');
        }
        // Perform raw SQL query to find locations within maxDistance using Haversine formula
        const locations = await this.prisma.$queryRaw `
      SELECT 
        l.*,
        (
          3958.8 * acos(
            LEAST(1.0, cos(radians(${coordinates.lat})) * 
            cos(radians(CAST(l.latitude AS float))) * 
            cos(radians(CAST(l.longitude AS float)) - radians(${coordinates.lng})) + 
            sin(radians(${coordinates.lat})) * 
            sin(radians(CAST(l.latitude AS float))))
          )
        ) as distance
      FROM "xplr"."locations" l
      WHERE 
        l.latitude IS NOT NULL 
        AND l.longitude IS NOT NULL
      ORDER BY distance ASC
    `;
        return locations;
    }
    async createLocation(data) {
        return this.prisma.location.create({
            data: {
                ...data,
                // Convert number to Decimal for latitude and longitude
                latitude: data.latitude ? new Prisma.Decimal(data.latitude) : null,
                longitude: data.longitude ? new Prisma.Decimal(data.longitude) : null,
            },
        });
    }
}
export default new LocationsService();
//# sourceMappingURL=location.service.js.map