import { Router } from 'express';
import { LocationsService } from '../services/location.service.js';
const router = Router();
const locationsService = new LocationsService();
const getAllLocations = async (req, res, next) => {
    try {
        const limit = req.query && req.query.limit ? parseInt(req.query.limit, 10) : 50;
        const offset = req.query && req.query.offset ? parseInt(req.query.offset, 10) : 0;
        const result = await locationsService.getAllLocations(limit, offset);
        res.json(result);
    }
    catch (error) {
        next(error);
    }
};
const getLocationById = async (req, res, next) => {
    try {
        const id = req.params && req.params.id ? Number(req.params.id) : NaN;
        if (isNaN(id)) {
            const error = new Error('Invalid location ID');
            error.statusCode = 400;
            return next(error);
        }
        const location = await locationsService.getLocationById(id);
        if (!location) {
            const error = new Error('Location not found');
            error.statusCode = 404;
            return next(error);
        }
        res.json(location);
    }
    catch (error) {
        ;
        error.message = 'Failed to fetch location: ' + error.message;
        next(error);
    }
};
const searchLocations = async (req, res, next) => {
    try {
        const { query, zipCode, type, hasStorePicks, offersSamples, sortBy, sortOrder, limit, offset, maxDistance, latitude, longitude, } = req.query || {};
        const searchParams = {
            query: query,
            zipCode: zipCode,
            type: type,
            hasStorePicks: hasStorePicks === 'true' ? true : hasStorePicks === 'false' ? false : undefined,
            offersSamples: offersSamples === 'true' ? true : offersSamples === 'false' ? false : undefined,
            sortBy: sortBy,
            sortOrder: sortOrder || 'asc',
            limit: limit ? parseInt(limit, 10) : undefined,
            offset: offset ? parseInt(offset, 10) : undefined,
            maxDistance: maxDistance ? parseInt(maxDistance, 10) : undefined,
            latitude: latitude ? parseFloat(latitude) : undefined,
            longitude: longitude ? parseFloat(longitude) : undefined,
        };
        const locations = await locationsService.searchLocations(searchParams);
        res.json(locations);
    }
    catch (error) {
        next(error);
    }
};
const createLocation = async (req, res, next) => {
    try {
        const { brandId, name, address, city, state, country, postalCode, latitude, longitude, googleMapsUrl, type } = req.body || {};
        if (!address || !city || !type) {
            const error = new Error('Required fields missing: address, city, and type are required');
            error.statusCode = 400;
            return next(error);
        }
        const validTypes = ['RETAIL', 'RESTAURANT', 'BAR', 'OTHER'];
        if (!validTypes.includes(type)) {
            const error = new Error(`Invalid location type. Must be one of: ${validTypes.join(', ')}`);
            error.statusCode = 400;
            return next(error);
        }
        const locationData = {
            brandId: brandId ? Number(brandId) : null,
            name,
            address,
            city,
            state,
            country,
            postalCode,
            latitude: latitude ? parseFloat(latitude) : null,
            longitude: longitude ? parseFloat(longitude) : null,
            googleMapsUrl,
            type: type,
        };
        const location = await locationsService.createLocation(locationData);
        res.status(201).json(location);
    }
    catch (error) {
        ;
        error.message = 'Failed to create location: ' + error.message;
        next(error);
    }
};
const getLatLongByZip = async (req, res, next) => {
    try {
        const zip = req.params && req.params.zip ? req.params.zip : undefined;
        if (!zip) {
            return res.status(400).json({ error: 'Zip code is required' });
        }
        const result = await locationsService.getLatLongByZipCode(zip);
        if (!result) {
            return res.status(404).json({ error: 'Zip code not found' });
        }
        res.json(result);
    }
    catch (error) {
        next(error);
    }
};
router.get('/', getAllLocations);
router.get('/search', searchLocations);
router.get('/zipcode/:zip', getLatLongByZip);
router.get('/:id', getLocationById);
router.post('/', createLocation);
export default router;
//# sourceMappingURL=location.routes.js.map