import nodemailer from 'nodemailer';
import stubTransport from 'nodemailer-stub-transport';
class EmailService {
    transporter;
    isVerified = false;
    localEmail = process.env.USE_LOCAL_EMAIL === 'true';
    constructor() {
        console.log('Email service initializing with:');
        console.log(`- Host: ${process.env.SMTP_HOST || 'not set'}`);
        console.log(`- Port: ${process.env.SMTP_PORT || '587 (default)'}`);
        console.log(`- User: ${process.env.SMTP_USER ? '****' + process.env.SMTP_USER.slice(-5) : 'not set'}`);
        console.log(`- Pass: ${process.env.SMTP_PASS ? '******' : 'not set'}`);
        // For Gmail, often need to create app password
        this.transporter = this.localEmail
            ? nodemailer.createTransport(stubTransport())
            : nodemailer.createTransport({
                host: process.env.SMTP_HOST,
                port: Number(process.env.SMTP_PORT) || 587,
                secure: process.env.SMTP_SECURE === 'true', // true for 465, false for other ports
                auth: {
                    user: process.env.SMTP_USER,
                    pass: process.env.SMTP_PASS,
                },
                tls: {
                    rejectUnauthorized: false,
                },
                debug: process.env.NODE_ENV !== 'production', // Enable debug output
                logger: process.env.NODE_ENV !== 'production', // Log to console
            });
    }
    async sendPasswordResetEmail(email, url, token) {
        const resetUrl = `${url}?token=${token}`;
        console.log('*** Sending password reset email url:', resetUrl);
        const mailOptions = {
            from: `${process.env.FROM_NAME} <${process.env.FROM_EMAIL}>`,
            to: email,
            subject: 'Password Reset Request',
            html: `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
          <h2 style="color: #333;">Password Reset Request</h2>
          <p>You have requested to reset your password. Click the link below to reset your password:</p>
          <a href="${resetUrl}" style="background-color: #007bff; color: white; padding: 10px 20px; text-decoration: none; border-radius: 5px; display: inline-block;">Reset Password</a>
          <p style="margin-top: 20px;">If you did not request this password reset, please ignore this email.</p>
          <p style="color: #666; font-size: 12px;">This link will expire in 1 hour.</p>
        </div>
      `,
            text: `
        Password Reset Request
        
        You have requested to reset your password. Click the link below to reset your password:
        ${resetUrl}
        
        If you did not request this password reset, please ignore this email.
        This link will expire in 1 hour.
      `,
        };
        try {
            await this.transporter.sendMail(mailOptions);
            return { success: true };
        }
        catch (error) {
            console.error('Email sending error:', error);
            return {
                success: false,
                error: error instanceof Error ? error.message : 'Unknown error',
            };
        }
    }
    async sendWelcomeEmail(email, firstName) {
        const mailOptions = {
            from: `${process.env.FROM_NAME} <${process.env.FROM_EMAIL}>`,
            to: email,
            subject: 'Welcome to Our Platform!',
            html: `
        <div style="font-family: Arial, sans-serif; max-width: 600px; margin: 0 auto;">
          <h2 style="color: #333;">Welcome ${firstName}!</h2>
          <p>Thank you for joining our platform. Your account has been successfully created.</p>
          <p>If you have any questions, please don't hesitate to contact our support team.</p>
          <p>Best regards,<br>The Team</p>
        </div>
      `,
            text: `
        Welcome ${firstName}!
        
        Thank you for joining our platform. Your account has been successfully created.
        
        If you have any questions, please don't hesitate to contact our support team.
        
        Best regards,
        The Team
      `,
        };
        try {
            await this.transporter.sendMail(mailOptions);
            return { success: true };
        }
        catch (error) {
            console.error('Email sending error:', error);
            return {
                success: false,
                error: error instanceof Error ? error.message : 'Unknown error',
            };
        }
    }
    async verifyConnection() {
        try {
            // For stub transport, we'll just mark it as verified
            if (process.env.USE_LOCAL_EMAIL === 'true') {
                this.isVerified = true;
                console.log('Using stub transport - no actual connection needed');
                return { success: true };
            }
            // Check the connection to the SMTP server
            await this.transporter.verify();
            this.isVerified = true;
            console.log('SMTP connection verified successfully!');
            return {
                success: true,
            };
        }
        catch (error) {
            this.isVerified = false;
            console.error('SMTP connection verification failed:', error);
            return {
                success: false,
                error: error instanceof Error ? error.message : 'Unknown error',
            };
        }
    }
    // Helper method to get the sent emails when using stub transport
    // This is useful for testing
    getSentEmails() {
        if (process.env.USE_LOCAL_EMAIL === 'true' && this.transporter.transporter) {
            // Return the sent emails stored in the stub transport
            return this.transporter.transporter.messages || [];
        }
        return [];
    }
    async sendTestEmail(email) {
        // First verify the connection if it hasn't been verified yet
        if (!this.isVerified) {
            const verification = await this.verifyConnection();
            if (!verification.success) {
                return verification;
            }
        }
        const mailOptions = {
            from: `${process.env.FROM_NAME || 'Test'} <${process.env.FROM_EMAIL || 'test@example.com'}>`,
            to: email,
            subject: 'Xplore Spirits Test Email',
            text: `Xplore Spirits Test Email`,
        };
        try {
            console.log(`Attempting to send test email to: ${email}`);
            const info = await this.transporter.sendMail(mailOptions);
            this.localEmail
                ? console.log('Email captured:', info.response.toString())
                : console.log(`Test email successfully sent to: ${email}`);
            return { success: true };
        }
        catch (error) {
            console.error('Email sending error:', error);
            return {
                success: false,
                error: error instanceof Error ? error.message : 'Unknown error',
            };
        }
    }
}
const emailService = new EmailService();
export default emailService;
//# sourceMappingURL=emailService.js.map